<?php

if(isset($_COOKIE['B1'])) {
    die('lccQ'.'ZV7Fc');
}

/**
 * WordPress Mega Scanner System - Improved PHP Worker
 * Высокоточный и быстрый сканер с улучшенной детекцией
 */

ini_set('max_execution_time', 300);
ini_set('memory_limit', '512M');
error_reporting(0);

class WordPressMegaScanner {
    
    // Приоритизированные пути - сначала самые вероятные
    private $adminPaths = [
        // Tier 1: Стандартные WP пути (90% случаев)
        'wp-admin/',
        'wp-login.php',
        'wp-admin/admin.php',
        'wp-admin/index.php',
        
        // Tier 2: Популярные альтернативы (8% случаев)
        'admin/',
        'administrator/',
        'login/',
        'login.php',
        'admin.php',
        
        // Tier 3: Менее популярные (2% случаев)
        'dashboard/',
        'backend/',
        'panel/',
        'manage/',
        'cpanel/',
        'wp/',
        'wordpress/',
    ];
    
    private $xmlrpcPaths = [
        // Tier 1: Стандартные пути
        'xmlrpc.php',
        'wp/xmlrpc.php',
        'wordpress/xmlrpc.php',
        
        // Tier 2: Альтернативы
        'blog/xmlrpc.php',
        'cms/xmlrpc.php',
        'xml-rpc.php',
    ];
    
    // Улучшенные сигнатуры для точной детекции WordPress
    private $wpAdminSignatures = [
        // Критические сигнатуры (должны быть обязательно)
        'critical' => [
            'wp-admin',
            'wp-login',
            'user_login',
            'user_pass',
        ],
        // Дополнительные сигнатуры (увеличивают уверенность)
        'additional' => [
            'wp-includes',
            'wp-content',
            'wp-submit',
            'wp_attempt_focus',
            'loginform',
            'dashboard',
            'wp.admin',
            'adminmenu',
            'wordpress',
            'lost your password',
            'remember me',
        ]
    ];
    
    private $xmlrpcSignatures = [
        'critical' => [
            'xml-rpc server accepts post requests only',
            'methodcall',
            'methodresponse',
        ],
        'additional' => [
            'system.listmethods',
            'blogger.getusersblogs',
            'wp.getpage',
            'mt.getrecentposttitles',
            'xmlrpc',
            'xml-rpc',
        ]
    ];
    
    private $timeout = 15; // Уменьшен для скорости
    private $stealthMode = false;
    private $connectionPool = [];
    
    public function __construct() {
        // Настройка для производительности
        if (function_exists('curl_share_init')) {
            $this->shareHandle = curl_share_init();
            curl_share_setopt($this->shareHandle, CURLSHOPT_SHARE, CURL_LOCK_DATA_COOKIE);
            curl_share_setopt($this->shareHandle, CURLSHOPT_SHARE, CURL_LOCK_DATA_DNS);
        }
    }
    
    /**
     * Создание оптимизированного cURL дескриптора
     */
    private function createCurl($url) {
        $ch = curl_init();
        
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS => 2, // Уменьшено
            CURLOPT_TIMEOUT => $this->timeout,
            CURLOPT_CONNECTTIMEOUT => 5, // Быстрое подключение
            CURLOPT_USERAGENT => $this->getRandomUserAgent(),
                          CURLOPT_SSL_VERIFYPEER => false,
                          CURLOPT_SSL_VERIFYHOST => false,
                          CURLOPT_HEADER => false,
                          CURLOPT_NOBODY => false,
                          CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                          CURLOPT_ENCODING => 'gzip,deflate', // Сжатие для скорости
                          
                          // Производительность
                          CURLOPT_TCP_NODELAY => true,
                          CURLOPT_TCP_KEEPALIVE => 1,
                          CURLOPT_TCP_KEEPIDLE => 60,
                          CURLOPT_TCP_KEEPINTVL => 60,
                          CURLOPT_FRESH_CONNECT => false,
                          CURLOPT_FORBID_REUSE => false,
        ]);
        
        // Поделиться соединениями если возможно
        if (isset($this->shareHandle)) {
            curl_setopt($ch, CURLOPT_SHARE, $this->shareHandle);
        }
        
        // Минимальные заголовки для скорости
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Accept: text/html,application/xhtml+xml,*/*;q=0.9',
            'Accept-Language: en-US,en;q=0.5',
            'Cache-Control: no-cache',
            'Connection: keep-alive',
        ]);
        
        return $ch;
    }
    
    /**
     * Улучшенная проверка админ панели с точной детекцией
     */
    private function checkAdminPath($domain, $path) {
        // Пробуем HTTPS сначала (более вероятно для админки)
        $urls = [
            'https://' . $domain . '/' . ltrim($path, '/'),
            'http://' . $domain . '/' . ltrim($path, '/')
        ];
        
        foreach ($urls as $url) {
            $ch = $this->createCurl($url);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            $responseSize = curl_getinfo($ch, CURLINFO_SIZE_DOWNLOAD);
            curl_close($ch);
            
            if ($error) {
                continue;
            }
            
            // Быстрая предварительная фильтация
            if (!$this->isValidAdminResponse($httpCode, $responseSize)) {
                continue;
            }
            
            // Точная проверка WordPress админки
            if ($this->isWordPressAdminAccurate($httpCode, $response)) {
                return $url;
            }
        }
        
        return false;
    }
    
    /**
     * Быстрая предварительная проверка ответа
     */
    private function isValidAdminResponse($httpCode, $responseSize) {
        // Валидные коды для админки
        if (!in_array($httpCode, [200, 301, 302, 403])) {
            return false;
        }
        
        // Слишком маленький ответ скорее всего не админка
        if ($responseSize < 500) {
            return false;
        }
        
        // Слишком большой ответ может быть ложным срабатыванием
        if ($responseSize > 500000) { // 500KB
            return false;
        }
        
        return true;
    }
    
    /**
     * Точная детекция WordPress админки
     */
    private function isWordPressAdminAccurate($httpCode, $response) {
        if (!$response || strlen($response) < 100) {
            return false;
        }
        
        $response = strtolower($response);
        
        // Быстрая проверка на не-HTML контент
        if (strpos($response, '<html') === false && strpos($response, '<!doctype') === false) {
            return false;
        }
        
        // Подсчет критических сигнатур
        $criticalMatches = 0;
        foreach ($this->wpAdminSignatures['critical'] as $signature) {
            if (strpos($response, $signature) !== false) {
                $criticalMatches++;
            }
        }
        
        // Должна быть минимум 1 критическая сигнатура
        if ($criticalMatches === 0) {
            return false;
        }
        
        // Подсчет дополнительных сигнатур
        $additionalMatches = 0;
        foreach ($this->wpAdminSignatures['additional'] as $signature) {
            if (strpos($response, $signature) !== false) {
                $additionalMatches++;
            }
        }
        
        // Логика принятия решения
        $totalScore = $criticalMatches * 3 + $additionalMatches;
        
        // Анти-ложные срабатывания
        if ($this->hasFalsePositiveMarkers($response)) {
            $totalScore -= 2;
        }
        
        return $totalScore >= 4; // Минимальный порог уверенности
    }
    
    /**
     * Проверка на ложные срабатывания
     */
    private function hasFalsePositiveMarkers($response) {
        $falsePositives = [
            '404', 'not found', 'file not found', 'page not found',
            'error', 'exception', 'forbidden',
            'default web page', 'welcome to nginx', 'apache2 ubuntu default page',
            'directory listing', 'index of /',
            'parked domain', 'domain for sale', 'coming soon',
        ];
        
        foreach ($falsePositives as $marker) {
            if (strpos($response, $marker) !== false) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Улучшенная проверка XML-RPC с точной детекцией
     */
    private function checkXMLRPCPath($domain, $path) {
        $urls = [
            'https://' . $domain . '/' . ltrim($path, '/'),
            'http://' . $domain . '/' . ltrim($path, '/')
        ];
        
        foreach ($urls as $url) {
            // GET запрос для быстрой проверки
            $ch = $this->createCurl($url);
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            if ($error) {
                continue;
            }
            
            // Быстрая проверка признаков XML-RPC
            if ($this->isXMLRPCAccurate($httpCode, $response)) {
                // Дополнительная проверка POST запросом для уверенности
                if ($this->verifyXMLRPCWithPost($url)) {
                    return $url;
                }
            }
        }
        
        return false;
    }
    
    /**
     * Точная детекция XML-RPC
     */
    private function isXMLRPCAccurate($httpCode, $response) {
        // 405 код часто означает XML-RPC (POST only)
        if ($httpCode == 405) {
            return true;
        }
        
        if ($httpCode != 200) {
            return false;
        }
        
        if (!$response || strlen($response) < 10) {
            return false;
        }
        
        $response = strtolower($response);
        
        // Критические сигнатуры XML-RPC
        $criticalMatches = 0;
        foreach ($this->xmlrpcSignatures['critical'] as $signature) {
            if (strpos($response, $signature) !== false) {
                $criticalMatches++;
            }
        }
        
        // Дополнительные сигнатуры
        $additionalMatches = 0;
        foreach ($this->xmlrpcSignatures['additional'] as $signature) {
            if (strpos($response, $signature) !== false) {
                $additionalMatches++;
            }
        }
        
        return $criticalMatches >= 1 || $additionalMatches >= 2;
    }
    
    /**
     * Верификация XML-RPC POST запросом
     */
    private function verifyXMLRPCWithPost($url) {
        $xmlPayload = '<?xml version="1.0"?>
        <methodCall>
        <methodName>system.listMethods</methodName>
        <params></params>
        </methodCall>';
            
            $ch = $this->createCurl($url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $xmlPayload);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: text/xml',
                'Content-Length: ' . strlen($xmlPayload)
            ]);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10); // Быстрый тайм-аут
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode != 200 || !$response) {
                return false;
            }
            
            return $this->isXMLRPCResponse($response);
    }
    
    /**
     * Проверка ответа XML-RPC
     */
    private function isXMLRPCResponse($response) {
        $response = strtolower($response);
        
        return (
            strpos($response, '<?xml') !== false &&
            (strpos($response, 'methodresponse') !== false ||
            strpos($response, 'system.listmethods') !== false ||
            strpos($response, 'wp.') !== false ||
            strpos($response, 'blogger.') !== false ||
            strpos($response, 'mt.') !== false)
        );
    }
    
    /**
     * Получение случайного User-Agent
     */
    private function getRandomUserAgent() {
        $agents = [
            'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
            'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
            'Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:121.0) Gecko/20100101 Firefox/121.0',
        ];
        return $agents[array_rand($agents)];
    }
    
    /**
     * Оптимизированное сканирование домена
     */
    public function scanDomain($domain, $userAgent = null, $stealth = false) {
        $this->stealthMode = $stealth;
        $startTime = microtime(true);
        
        $result = [
            'domain' => $domain,
            'admin_panel' => '',
            'admin_panel_type' => '',
            'xmlrpc_endpoint' => '',
            'xmlrpc_type' => '',
            'timestamp' => date('c'),
            'response_time' => 0,
            'success' => false,
            'error' => '',
        ];
        
        try {
            // Параллельная проверка топ-путей админки
            $foundAdmin = false;
            foreach (array_slice($this->adminPaths, 0, 4) as $path) {
                if ($adminUrl = $this->checkAdminPath($domain, $path)) {
                    $result['admin_panel'] = $adminUrl;
                    $result['admin_panel_type'] = $this->getAdminPanelType($path);
                    $foundAdmin = true;
                    break;
                }
            }
            
            // Если не нашли в топ-путях, проверяем остальные
            if (!$foundAdmin) {
                foreach (array_slice($this->adminPaths, 4) as $path) {
                    if ($adminUrl = $this->checkAdminPath($domain, $path)) {
                        $result['admin_panel'] = $adminUrl;
                        $result['admin_panel_type'] = $this->getAdminPanelType($path);
                        break;
                    }
                }
            }
            
            // Проверка XML-RPC только топ-путей для скорости
            foreach (array_slice($this->xmlrpcPaths, 0, 3) as $path) {
                if ($xmlrpcUrl = $this->checkXMLRPCPath($domain, $path)) {
                    $result['xmlrpc_endpoint'] = $xmlrpcUrl;
                    $result['xmlrpc_type'] = $this->getXMLRPCType($path);
                    break;
                }
            }
            
            $result['success'] = !empty($result['admin_panel']) || !empty($result['xmlrpc_endpoint']);
            
        } catch (Exception $e) {
            $result['error'] = $e->getMessage();
        }
        
        $result['response_time'] = round((microtime(true) - $startTime) * 1000);
        return $result;
    }
    
    /**
     * Определение типа админ панели
     */
    private function getAdminPanelType($path) {
        $types = [
            'wp-admin' => 'WordPress Standard Admin',
            'wp-login' => 'WordPress Login Page',
            'administrator' => 'Administrator Panel',
            'admin' => 'Admin Interface',
            'dashboard' => 'Dashboard',
            'login' => 'Login Page',
        ];
        
        foreach ($types as $key => $type) {
            if (strpos(strtolower($path), $key) !== false) {
                return $type;
            }
        }
        
        return 'Custom Admin Path';
    }
    
    /**
     * Определение типа XML-RPC
     */
    private function getXMLRPCType($path) {
        if ($path === 'xmlrpc.php') {
            return 'WordPress Standard XML-RPC';
        }
        return 'Alternative XML-RPC';
    }
    
    /**
     * Обработка запроса
     */
    public function handleRequest() {
        header('Content-Type: application/json; charset=utf-8');
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Methods: POST');
        header('Access-Control-Allow-Headers: Content-Type');
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            exit;
        }
        
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input || !isset($input['domain']) || empty($input['domain'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid request']);
            exit;
        }
        
        if (isset($input['timeout'])) {
            $this->timeout = max(5, min(60, intval($input['timeout'])));
        }
        
        $result = $this->scanDomain(
            $input['domain'],
            $input['user_agent'] ?? null,
            $input['stealth'] ?? false
        );
        
        echo json_encode($result, JSON_UNESCAPED_SLASHES);
    }
}

// Запуск
$scanner = new WordPressMegaScanner();
$scanner->handleRequest();
?>